/*
 * Copyright (C) filoghost and contributors
 *
 * SPDX-License-Identifier: GPL-3.0-or-later
 */
package me.filoghost.holographicdisplays.plugin.config;

import me.filoghost.fcommons.collection.CollectionUtils;
import me.filoghost.fcommons.config.BaseConfigManager;
import me.filoghost.fcommons.config.Config;
import me.filoghost.fcommons.config.ConfigErrors;
import me.filoghost.fcommons.config.ConfigLoader;
import me.filoghost.fcommons.config.ConfigPath;
import me.filoghost.fcommons.config.ConfigSection;
import me.filoghost.fcommons.config.ConfigType;
import me.filoghost.fcommons.config.FileConfig;
import me.filoghost.fcommons.config.exception.ConfigException;
import me.filoghost.fcommons.config.exception.ConfigLoadException;
import me.filoghost.fcommons.config.exception.ConfigSaveException;
import me.filoghost.fcommons.config.exception.ConfigValueException;
import me.filoghost.fcommons.config.mapped.MappedConfigLoader;
import me.filoghost.fcommons.logging.ErrorCollector;
import me.filoghost.fcommons.logging.Log;
import me.filoghost.holographicdisplays.plugin.format.DisplayFormat;
import me.filoghost.holographicdisplays.plugin.internal.hologram.InternalHologram;
import me.filoghost.holographicdisplays.plugin.internal.placeholder.AnimationPlaceholder;
import me.filoghost.holographicdisplays.plugin.internal.placeholder.AnimationPlaceholderFactory;
import me.filoghost.holographicdisplays.plugin.util.FileUtils;

import java.io.IOException;
import java.nio.file.Files;
import java.nio.file.Path;
import java.util.ArrayList;
import java.util.Collections;
import java.util.HashMap;
import java.util.List;
import java.util.Map;
import java.util.Map.Entry;
import java.util.stream.Stream;

public class ConfigManager extends BaseConfigManager {

    private final MappedConfigLoader<SettingsModel> mainConfigLoader;
    private final ConfigLoader databaseConfigLoader;
    private final ConfigLoader staticReplacementsConfigLoader;

    public ConfigManager(Path rootDataFolder) {
        super(rootDataFolder);
        this.mainConfigLoader = getMappedConfigLoader("config.yml", SettingsModel.class);
        this.databaseConfigLoader = getConfigLoader("database.yml");
        this.staticReplacementsConfigLoader = getConfigLoader("custom-placeholders.yml");
    }

    public void reloadMainSettings(ErrorCollector errorCollector) {
        SettingsModel mainConfig;

        try {
            mainConfig = mainConfigLoader.init();
        } catch (ConfigException e) {
            logConfigException(errorCollector, mainConfigLoader.getFile(), e);
            mainConfig = new SettingsModel(); // Fallback: use default values
        }

        Settings.load(mainConfig, errorCollector);
    }

    public List<InternalHologramConfig> readHologramDatabase(ErrorCollector errorCollector) {
        Config config;
        try {
            config = databaseConfigLoader.init();
        } catch (ConfigException e) {
            logConfigException(errorCollector, databaseConfigLoader.getFile(), e);
            return Collections.emptyList();
        }

        List<InternalHologramConfig> hologramConfigs = new ArrayList<>();

        for (Entry<ConfigPath, ConfigSection> entry : config.toMap(ConfigType.SECTION).entrySet()) {
            String hologramName = entry.getKey().asRawKey();
            ConfigSection hologramConfigSection = entry.getValue();
            hologramConfigs.add(new InternalHologramConfig(hologramName, hologramConfigSection));
        }

        return hologramConfigs;
    }

    public void saveHologramDatabase(List<InternalHologram> holograms) {
        Config config = new Config();
        config.setHeader(
                "",
                "Please do NOT edit this file manually if possible.",
                ""
        );

        for (InternalHologram hologram : holograms) {
            InternalHologramConfig hologramConfig = new InternalHologramConfig(hologram);
            config.setConfigSection(hologram.getName(), hologramConfig.getSerializedConfigSection());
        }

        try {
            databaseConfigLoader.save(config);
        } catch (ConfigException e) {
            Log.severe("Error while saving holograms database file \"" + formatPath(databaseConfigLoader.getFile()) + "\"", e);
        }
    }

    public AnimationPlaceholderFactory loadAnimations(ErrorCollector errorCollector) {
        Map<String, AnimationPlaceholder> animationsByFileName = new HashMap<>();
        Path animationsFolder = getAnimationsFolder();

        try {
            if (!Files.isDirectory(animationsFolder)) {
                Files.createDirectories(animationsFolder);
                try {
                    Path exampleAnimationFile = animationsFolder.resolve("example.yml");
                    getConfigLoader(exampleAnimationFile).createDefault();
                } catch (ConfigSaveException e) {
                    errorCollector.add(e, "could not add example animation file");
                }
            }

            try (Stream<Path> animationFiles = Files.list(animationsFolder)) {
                animationFiles.filter(this::isYamlFile).forEach(file -> {
                    try {
                        String fileName = file.getFileName().toString();
                        AnimationPlaceholder animationPlaceholder = loadAnimationPlaceholder(file);
                        animationsByFileName.put(fileName, animationPlaceholder);
                    } catch (ConfigException e) {
                        logConfigException(errorCollector, file, e);
                    }
                });
            }
        } catch (IOException e) {
            errorCollector.add(e, "error loading animation files");
        }

        return new AnimationPlaceholderFactory(animationsByFileName);
    }

    private AnimationPlaceholder loadAnimationPlaceholder(Path animationFile) throws ConfigLoadException, ConfigValueException {
        AnimationConfig animationConfig = new AnimationConfig(getConfigLoader(animationFile).load());
        return new AnimationPlaceholder(
                animationConfig.getIntervalTicks(),
                CollectionUtils.toArrayList(animationConfig.getFrames(), DisplayFormat::apply)
        );
    }

    public void reloadStaticReplacements(ErrorCollector errorCollector) {
        FileConfig staticReplacementsConfig;

        try {
            staticReplacementsConfig = staticReplacementsConfigLoader.init();
        } catch (ConfigException e) {
            logConfigException(errorCollector, staticReplacementsConfigLoader.getFile(), e);
            staticReplacementsConfig = new FileConfig(staticReplacementsConfigLoader.getFile()); // Fallback: empty config
        }

        StaticReplacements.load(staticReplacementsConfig, errorCollector);
    }

    public Path getAnimationsFolder() {
        return getRootDataFolder().resolve("animations");
    }

    private boolean isYamlFile(Path file) {
        if (!Files.isRegularFile(file)) {
            return false;
        }

        return FileUtils.hasFileExtension(file, "yml", "yaml");
    }

    private void logConfigException(ErrorCollector errorCollector, Path file, ConfigException e) {
        errorCollector.add(e, "error while loading config file \"" + formatPath(file) + "\"");
    }

    public String formatPath(Path path) {
        return ConfigErrors.formatPath(getRootDataFolder(), path);
    }

}
