/**
 * Fabled
 * studio.magemonkey.fabled.listener.CastItemListener
 * <p>
 * The MIT License (MIT)
 * <p>
 * Copyright (c) 2024 MageMonkeyStudio
 * <p>
 * Permission is hereby granted, free of charge, to any person obtaining a copy
 * of this software and associated documentation files (the "Software"), to deal
 * in the Software without restriction, including without limitation the rights
 * to use, copy, modify, merge, publish, distribute, sublicense, and/or sell
 * copies of the Software, and to permit persons to whom the Software is
 * furnished to do so, subject to the following conditions:
 * <p>
 * The above copyright notice and this permission notice shall be included in
 * all copies or substantial portions of the Software.
 * <p>
 * THE SOFTWARE IS PROVIDED "AS IS", WITHOUT WARRANTY OF ANY KIND, EXPRESS OR
 * IMPLIED, INCLUDING BUT NOT LIMITED TO THE WARRANTIES OF MERCHANTABILITY,
 * FITNESS FOR A PARTICULAR PURPOSE AND NONINFRINGEMENT. IN NO EVENT SHALL THE
 * AUTHORS OR COPYRIGHT HOLDERS BE LIABLE FOR ANY CLAIM, DAMAGES OR OTHER
 * LIABILITY, WHETHER IN AN ACTION OF CONTRACT, TORT OR OTHERWISE, ARISING FROM,
 * OUT OF OR IN CONNECTION WITH THE SOFTWARE OR THE USE OR OTHER DEALINGS IN
 * THE SOFTWARE.
 */
package studio.magemonkey.fabled.listener;

import org.bukkit.Bukkit;
import org.bukkit.Material;
import org.bukkit.entity.Player;
import org.bukkit.event.EventHandler;
import org.bukkit.event.EventPriority;
import org.bukkit.event.block.Action;
import org.bukkit.event.entity.PlayerDeathEvent;
import org.bukkit.event.inventory.InventoryAction;
import org.bukkit.event.inventory.InventoryClickEvent;
import org.bukkit.event.inventory.InventoryType;
import org.bukkit.event.player.*;
import org.bukkit.inventory.ItemStack;
import org.bukkit.inventory.PlayerInventory;
import org.bukkit.scheduler.BukkitRunnable;
import studio.magemonkey.fabled.Fabled;
import studio.magemonkey.fabled.api.event.PlayerClassChangeEvent;
import studio.magemonkey.fabled.api.event.PlayerSkillUnlockEvent;
import studio.magemonkey.fabled.api.player.PlayerData;
import studio.magemonkey.fabled.api.player.PlayerSkillSlot;

import java.util.*;

/**
 * Handles the alternate casting option for casting via a cycling slot
 */
public class CastItemListener extends FabledListener {
    private final Map<UUID, PlayerSkillSlot> data            = new HashMap<>();
    private final Set<UUID>                  playersDropping = new HashSet<>();

    private void cleanup(Player player) {
        data.remove(player.getUniqueId());
        if (Fabled.getSettings().isWorldEnabled(player.getWorld()))
            player.getInventory().setItem(Fabled.getSettings().getCastSlot(), null);
    }

    @Override
    public void init() {
        MainListener.registerJoin(this::init);
        MainListener.registerClear(this::handleClear);
        for (Player player : Bukkit.getOnlinePlayers())
            init(player);
    }

    /**
     * Cleans up the listener functions
     */
    @Override
    public void cleanup() {
        for (Player player : Bukkit.getOnlinePlayers())
            cleanup(player);
    }

    /**
     * Re-initializes cast data on class change
     *
     * @param event event details
     */
    @EventHandler
    public void onClassChange(PlayerClassChangeEvent event) {
        data.get(event.getPlayerData().getPlayer().getUniqueId()).init(event.getPlayerData());
    }

    /**
     * Enables/disables cast when changing worlds
     *
     * @param event event details
     */
    @EventHandler(priority = EventPriority.LOWEST)
    public void onWorldChangePre(PlayerChangedWorldEvent event) {
        boolean from = Fabled.getSettings().isWorldEnabled(event.getFrom());
        boolean to   = Fabled.getSettings().isWorldEnabled(event.getPlayer().getWorld());
        if (from && !to)
            event.getPlayer().getInventory().setItem(Fabled.getSettings().getCastSlot(), null);
    }

    @EventHandler(priority = EventPriority.MONITOR)
    public void onWorldChange(PlayerChangedWorldEvent event) {
        boolean from = Fabled.getSettings().isWorldEnabled(event.getFrom());
        boolean to   = Fabled.getSettings().isWorldEnabled(event.getPlayer().getWorld());

        if (to && !from) init(event.getPlayer());
    }

    private PlayerSkillSlot get(Player player) {
        return data.get(player.getUniqueId());
    }

    private PlayerSkillSlot get(PlayerData data) {
        return this.data.get(data.getPlayer().getUniqueId());
    }

    /**
     * Gives the player the cast item
     *
     * @param player player to give to
     */
    private void init(Player player) {
        if (!Fabled.getSettings().isWorldEnabled(player.getWorld())) return;

        PlayerSkillSlot slotData = new PlayerSkillSlot();
        data.put(player.getUniqueId(), slotData);
        slotData.init(Fabled.getData(player));

        PlayerInventory inv  = player.getInventory();
        int             slot = Fabled.getSettings().getCastSlot();
        ItemStack       item = inv.getItem(slot);
        slotData.updateItem(player);
        if (item != null && item.getType() != Material.AIR)
            inv.addItem(item);
    }

    /**
     * Removes the cast item on quit
     *
     * @param event event details
     */

    @EventHandler
    public void onQuit(PlayerQuitEvent event) {
        cleanup(event.getPlayer());
    }

    /**
     * Adds unlocked skills to the skill bar if applicable
     *
     * @param event event details
     */
    @EventHandler
    public void onUnlock(PlayerSkillUnlockEvent event) {
        get(event.getPlayerData()).unlock(event.getUnlockedSkill());
    }

    /**
     * Prevents moving the cast item
     *
     * @param event event details
     */
    @EventHandler
    public void onClick(InventoryClickEvent event) {
        if (Fabled.getSettings().isWorldEnabled(event.getWhoClicked().getWorld())) {
            if (event.getSlot() == Fabled.getSettings().getCastSlot()
                    && event.getSlotType() == InventoryType.SlotType.QUICKBAR)
                event.setCancelled(true);
            else if (event.getAction() == InventoryAction.HOTBAR_SWAP
                    && event.getHotbarButton() == Fabled.getSettings().getCastSlot())
                event.setCancelled(true);
        }
    }

    /**
     * Casts a skill when dropping the cast item
     *
     * @param event event details
     */
    @EventHandler
    public void onDrop(PlayerDropItemEvent event) {
        if (Fabled.getSettings().isWorldEnabled(event.getPlayer().getWorld())
                && event.getPlayer().getInventory().getHeldItemSlot() == Fabled.getSettings().getCastSlot()) {
            event.setCancelled(true);
            get(event.getPlayer()).activate();
            this.playersDropping.add(event.getPlayer().getUniqueId());
            new BukkitRunnable() {
                @Override
                public void run() {
                    playersDropping.remove(event.getPlayer().getUniqueId());
                }
            }.runTask(Fabled.inst());
        }
    }

    @EventHandler
    public void onDeath(PlayerDeathEvent event) {
        if (Fabled.getSettings().isWorldEnabled(event.getEntity().getWorld())) {
            event.getDrops().remove(event.getEntity().getInventory().getItem(Fabled.getSettings().getCastSlot()));
        }
    }

    /**
     * Cycles through skills upon interact
     *
     * @param event event details
     */
    @EventHandler
    public void onInteract(PlayerInteractEvent event) {
        // Cycling skills
        if (Fabled.getSettings().isWorldEnabled(event.getPlayer().getWorld())
                && event.getPlayer().getInventory().getHeldItemSlot() == Fabled.getSettings().getCastSlot()) {
            event.setCancelled(true);
            if (event.getAction() == Action.LEFT_CLICK_AIR || event.getAction() == Action.LEFT_CLICK_BLOCK) {
                if (this.playersDropping.remove(event.getPlayer().getUniqueId())) {
                    return;
                }
                get(event.getPlayer()).prev();
            } else if (event.getAction() == Action.RIGHT_CLICK_AIR || event.getAction() == Action.RIGHT_CLICK_BLOCK)
                get(event.getPlayer()).next();
        }
    }

    @EventHandler
    public void onItemHeld(PlayerItemHeldEvent event) {
        data.get(event.getPlayer().getUniqueId())
                .setHovering(event.getNewSlot() == Fabled.getSettings().getCastSlot());
    }

    private void handleClear(final Player player) {
        player.getInventory().setItem(Fabled.getSettings().getCastSlot(), Fabled.getSettings().getCastItem());
    }
}
