/**
 * Fabled
 * studio.magemonkey.fabled.cmd.CmdUnbind
 * <p>
 * The MIT License (MIT)
 * <p>
 * Copyright (c) 2024 MageMonkeyStudio
 * <p>
 * Permission is hereby granted, free of charge, to any person obtaining a copy
 * of this software and associated documentation files (the "Software") to deal
 * in the Software without restriction, including without limitation the rights
 * to use, copy, modify, merge, publish, distribute, sublicense, and/or sell
 * copies of the Software, and to permit persons to whom the Software is
 * furnished to do so, subject to the following conditions:
 * <p>
 * The above copyright notice and this permission notice shall be included in
 * all copies or substantial portions of the Software.
 * <p>
 * THE SOFTWARE IS PROVIDED "AS IS", WITHOUT WARRANTY OF ANY KIND, EXPRESS OR
 * IMPLIED, INCLUDING BUT NOT LIMITED TO THE WARRANTIES OF MERCHANTABILITY,
 * FITNESS FOR A PARTICULAR PURPOSE AND NONINFRINGEMENT. IN NO EVENT SHALL THE
 * AUTHORS OR COPYRIGHT HOLDERS BE LIABLE FOR ANY CLAIM, DAMAGES OR OTHER
 * LIABILITY, WHETHER IN AN ACTION OF CONTRACT, TORT OR OTHERWISE, ARISING FROM,
 * OUT OF OR IN CONNECTION WITH THE SOFTWARE OR THE USE OR OTHER DEALINGS IN
 * THE SOFTWARE.
 */
package studio.magemonkey.fabled.cmd;

import org.bukkit.Material;
import org.bukkit.command.CommandSender;
import org.bukkit.entity.Player;
import org.bukkit.inventory.ItemStack;
import org.bukkit.plugin.Plugin;
import studio.magemonkey.codex.mccore.commands.ConfigurableCommand;
import studio.magemonkey.codex.mccore.commands.IFunction;
import studio.magemonkey.codex.mccore.util.TextFormatter;
import studio.magemonkey.fabled.Fabled;
import studio.magemonkey.fabled.api.player.PlayerData;
import studio.magemonkey.fabled.language.RPGFilter;
import studio.magemonkey.fabled.listener.BindListener;

import java.util.List;
import java.util.stream.Collectors;

/**
 * Command to bind a skill to an item
 */
public class CmdUnbind implements IFunction {
    private static final String NOT_PLAYER  = "not-player";
    private static final String NOT_BOUND   = "not-bound";
    private static final String NO_ITEM     = "no-item";
    private static final String SKILL_BOUND = "skill-unbound";
    private static final String DISABLED    = "world-disabled";

    /**
     * Executes the command
     *
     * @param command owning command
     * @param plugin  plugin reference
     * @param sender  sender of the command
     * @param args    arguments
     * @param silent
     */
    @Override
    public void execute(ConfigurableCommand command, Plugin plugin, CommandSender sender, String[] args, boolean silent) {
        if (!(sender instanceof Player)) {
            command.sendMessage(sender, NOT_PLAYER, "&4Only players can use this command", silent);
        }

        // Disabled world
        else if (!Fabled.getSettings().isWorldEnabled(((Player) sender).getWorld())) {
            command.sendMessage(sender, DISABLED, "&4You cannot use this command in this world", silent);
        } else {
            ItemStack item = ((Player) sender).getInventory().getItemInMainHand();
            if (item == null || item.getType() == Material.AIR) {
                command.sendMessage(sender, NO_ITEM, "&4You are not holding an item", silent);
                return;
            }

            PlayerData   player = Fabled.getData((Player) sender);
            List<String> bound  = BindListener.getBoundSkills(item);
            if (bound.isEmpty()) {
                command.sendMessage(sender, NOT_BOUND, "&4There are no skills bound to the held item", silent);
            } else {
                BindListener.setBoundSkills(item, List.of());
                bound = bound.stream()
                        .map(skillName -> player.getSkill(skillName).getData().getName())
                        .collect(Collectors.toList());
                command.sendMessage(sender,
                        SKILL_BOUND,
                        "&6{skill} &2has been unbound from &6{item}",
                        silent, RPGFilter.SKILL.setReplacement(String.join(", ", bound)),
                        RPGFilter.ITEM.setReplacement(TextFormatter.format(item.getType().name())));
            }
        }
    }
}