/**
 * Codex
 * studio.magemonkey.codex.mccore.chat.Chat
 * <p>
 * The MIT License (MIT)
 * <p>
 * Copyright (c) 2024 MageMonkeyStudio
 * <p>
 * Permission is hereby granted, free of charge, to any person obtaining a copy
 * of this software and associated documentation files (the "Software") to deal
 * in the Software without restriction, including without limitation the rights
 * to use, copy, modify, merge, publish, distribute, sublicense, and/or sell
 * copies of the Software, and to permit persons to whom the Software is
 * furnished to do so, subject to the following conditions:
 * <p>
 * The above copyright notice and this permission notice shall be included in
 * all copies or substantial portions of the Software.
 * <p>
 * THE SOFTWARE IS PROVIDED "AS IS", WITHOUT WARRANTY OF ANY KIND, EXPRESS OR
 * IMPLIED, INCLUDING BUT NOT LIMITED TO THE WARRANTIES OF MERCHANTABILITY,
 * FITNESS FOR A PARTICULAR PURPOSE AND NONINFRINGEMENT. IN NO EVENT SHALL THE
 * AUTHORS OR COPYRIGHT HOLDERS BE LIABLE FOR ANY CLAIM, DAMAGES OR OTHER
 * LIABILITY, WHETHER IN AN ACTION OF CONTRACT, TORT OR OTHERWISE, ARISING FROM,
 * OUT OF OR IN CONNECTION WITH THE SOFTWARE OR THE USE OR OTHER DEALINGS IN
 * THE SOFTWARE.
 */
package studio.magemonkey.codex.mccore.chat;

import org.bukkit.Bukkit;
import org.bukkit.Location;
import org.bukkit.entity.Player;
import studio.magemonkey.codex.CodexEngine;
import studio.magemonkey.codex.mccore.config.Config;

import java.util.Hashtable;

/**
 * Main helper method for the chat resources including accessing
 * player data and sending messages to target groups
 */
public class Chat {

    static Hashtable<String, ChatData> players = new Hashtable<>();

    /**
     * Retrieves the player with the given name
     *
     * @param playerName name of the player
     * @return chat data associated with the player
     */
    public static ChatData getPlayerData(String playerName) {

        playerName = playerName.toLowerCase();

        // Initialize data if it doesn't exist
        if (!players.containsKey(playerName)) {
            CodexEngine core       = CodexEngine.get();
            Config      configFile = core.getConfigFile(core, "data");
            ChatData    data       = new ChatData(configFile.getConfig(), playerName);
            configFile.addSavable(data, playerName + ".");
            players.put(playerName, data);
        }

        return players.get(playerName);
    }

    /**
     * Unlocks the prefix for all players
     *
     * @param prefix the prefix to unlock
     * @param apply  whether or not to automatically apply it
     */
    public static void unlockPrefix(Prefix prefix, boolean apply) {
        for (ChatData data : players.values()) data.unlockPrefix(prefix, apply);
    }

    /**
     * Removes the prefix for all players
     *
     * @param pluginName name of the plugin with the prefix
     * @param prefix     the text of the prefix (with or without color)
     */
    public static void removePrefix(String pluginName, String prefix) {
        for (ChatData data : players.values()) data.removePrefix(pluginName, prefix);
    }

    /**
     * Clears the plugin prefix for all players
     *
     * @param pluginName name of the plugin with the prefix
     */
    public static void clearPluginPrefixes(String pluginName) {
        for (ChatData data : players.values()) data.clearPluginPrefix(pluginName);
    }

    /**
     * Sends a message to all players with the given permission
     *
     * @param permission permission required to get the message
     * @param message    message to send
     */
    public static void sendMessage(String permission, String message) {
        for (Player player : Bukkit.getOnlinePlayers()) {
            if (player.hasPermission(permission)) player.sendMessage(message);
        }
    }

    /**
     * Sends a message to all players within the defined cuboid
     *
     * @param point1  the first point
     * @param point2  the second point
     * @param message the message to be sent
     */
    public static void sendMessage(Location point1, Location point2, String message) {
        if (point1.getWorld() != point2.getWorld()) return;
        sendMessage(point1, point2.getBlockX() - point1.getBlockX(), point2.getBlockY() - point1.getBlockY(),
                point2.getBlockZ() - point1.getBlockZ(), message);
    }

    /**
     * Sends a message to all players within the defined cuboid
     *
     * @param point   the initial point
     * @param width   width of the cuboid (x direction)
     * @param height  height of the cuboid (y direction)
     * @param depth   depth of the cuboid (z direction)
     * @param message message to be sent
     */
    public static void sendMessage(Location point, int width, int height, int depth, String message) {
        if (width < 0) point.setX(point.getX() + width);
        if (height < 0) point.setY(point.getY() + height);
        if (depth < 0) point.setZ(point.getZ() + depth);
        for (Player player : Bukkit.getOnlinePlayers()) {
            Location loc = player.getLocation();
            if (loc.getX() >= point.getX() && loc.getY() >= point.getY() && loc.getZ() >= point.getZ()
                    && loc.getX() <= point.getX() + Math.abs(width) && loc.getY() <= point.getY() + Math.abs(height)
                    && loc.getZ() <= point.getZ() + Math.abs(depth))
                player.sendMessage(message);
        }
    }

    /**
     * Sends a message to all players within the defined sphere or cylinder
     *
     * @param center the center of the region
     * @param radius radius of the sphere/cylinder
     * @param sphere sphere if true, cylinder if false (cylinder contains all y within the defined circle)
     */
    public static void sendMessage(Location center, int radius, boolean sphere, String message) {
        for (Player player : Bukkit.getOnlinePlayers()) {
            Location loc = player.getLocation();
            if (!sphere) loc.setY(center.getY());
            if (loc.distanceSquared(center) < radius * radius) player.sendMessage(message);
        }
    }
}
